#include <lvsmon.h>
#include <config.h>
#include <util.h>

#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <netdb.h>
#include <errno.h>

static const char rcsid[]=
	"@(#) $Id: config.c,v 1.5 2002/01/24 12:38:47 gianni Exp $";

void config_add_mon(p_mon m)
{
	/* validate the security critical fields */
	m->s=-1;
	m->next=NULL;

	if ( monitors && last_mon ) {
		last_mon->next=m;
		last_mon=m;
	}else{
		monitors=m;
		last_mon=m;
	}

	num_monitors++;
}

p_plugin find_plugin(char *name)
{
	p_plugin p;

	for(p=globals.plugins; p; p=p->next) {
		if ( strcmp(name, p->name) == 0 ) return p;
	}
	return NULL;
}

int config_resolve(char *ip, in_addr_t *in)
{
	struct hostent *h;
	struct in_addr a;

	if ( inet_aton(ip, &a) ) {
		*in=a.s_addr;
		return 1;
	}

	if ( (h=gethostbyname(ip)) ) {
		memcpy(&a, h->h_addr, h->h_length);
		*in=a.s_addr;
		return 1;
	}
				
	return 0;
}

/* Parse an individual line */
int config_line(const char *fn, int line, char *ln, int llen)
{
	char *m=NULL;
	p_mon tmp=NULL;
	char *ip, *port, *priv;
	
	/* Buffer the text and add trailing NULL */
	if ( !(m=malloc(llen+1)) || !(tmp=malloc(sizeof(mon_t))) ) 
	{
		fprintf(stderr, "%s: %s:%i: Aborting! Out of memory!\n",
			globals.cmd, fn, line);
		die_err();
	}
	memset(tmp, 0, sizeof(mon_t));
	memcpy(m, ln, llen);
	m[llen]='\0';

	/* Parse the line */
	if ( (ip=strchr(m, ':')) ) {
		*ip='\0';
		ip++;
	}else goto die_parse;
	
	if ( (port=strchr(ip, ':')) ) {
		*port='\0';
		port++;
	}else goto die_parse;

	if ( (priv=strchr(port, ':')) ) {
		*priv='\0';
		priv++;
	}else priv=NULL;

	/* Get the port number */
	if ( !(tmp->port=atoi(port)) ) {
		goto die_parse;
	}
	
	/* Find the plugin */
	if ( !(tmp->plugin=find_plugin(m)) ) {
		fprintf(stderr, "%s: %s:%i: %s plugin not installed\n",
			globals.cmd, fn, line, m);
		goto die;
	}

	/* Validate that shit */
	if ( !tmp->plugin->validate(priv, &tmp->priv) ) {
		goto die_parse;
	}
	
	/* Resolve the IP address */
	if ( !config_resolve(ip, &tmp->addr)) {
		fprintf(stderr, "%s: %s:%i: Unable to resolve address: %s\n",
			globals.cmd, fn, line, ip);
		goto die;
	}

	/* Add it */
	tmp->interval=globals.interval;
	config_add_mon(tmp);
	
	/* Free the relevent stuff */
	free(m);
	return 1;

die_parse:
	fprintf(stderr, "%s: %s:%i: Parse error\n",
		globals.cmd, fn, line);
die:
	if (tmp) free(tmp);
	if (m) free(m);
	return 0;
}

/* Parse line by line */
int config_go(const char *fn, char *map, int len)
{
	int ret=1;
	char *cur;
	char *begin=map;
	int llen; /* line length */
	int lines=0;

	for(cur=map; (long)cur<((long)map)+len; cur++ )
	{
		if ( *cur=='\n' ) {
			lines++;
			llen=((long)cur)-((long)begin);
			if ( *begin!='#' && llen ) {
				if (!config_line(fn, lines, begin, llen)) {
					ret=0;
					break;
				}
			}
			begin=cur+1;
		}
	}
	return ret;
}

int config_parse(char *fn)
{
	int fd;
	char *map;
	struct stat st;
	int ret=0;

	/* Open the file an memory map it */
	if ( (fd=open(fn, O_RDONLY))<0 ) {
		fprintf(stderr, "%s: %s: Can't open config file: %s\n",
			globals.cmd, fn, get_err());
		return ret;
	}

	if ( (fstat(fd, &st)) < 0 ) {
		fprintf(stderr, "%s: %s: Can't determine size of config file: %s\n",
			globals.cmd, fn, get_err());
		close(fd);
		return ret;
	}

	if ( (map=mmap(NULL, st.st_size, PROT_READ, MAP_SHARED, fd, 0))==MAP_FAILED )
	{
		fprintf(stderr, "%s: %s: Can't memory map config file: %s\n",
			globals.cmd, fn, get_err());
		close(fd);
		return ret;
	}

	ret=config_go(fn, map, st.st_size);
	
	munmap(map, st.st_size);
	close(fd);
	return ret;
}
