#include <lvsmon.h>
#include <main.h>
#include <config.h>
#include <util.h>
#include <status.h>

/* Plugins */
#include <http.h>
#include <banner.h>

#include <asyncio.h>
#include <sys/socket.h>
#include <sys/poll.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <arpa/inet.h>
#include <getopt.h>

static const char rcsid[]=
	"@(#) $Id: main.c,v 1.7 2002/01/24 12:38:47 gianni Exp $";

/* Forward declare phase handlers  */
static void phase_connect(p_mon);
static inline void phase_conresult(p_mon);
static void phase_query(p_mon);
static inline void phase_recv(p_mon);
static void phase_wait(p_mon, int);

/* Global configuration variables */
globals_t globals;

/* List of monitors */
p_mon monitors=NULL;
p_mon last_mon=NULL;
int num_monitors=0;

/* Big poll set */
struct pollfd *pf=NULL;
int count; /* never higher than num_monitors */

struct option opts[]=
{
	{"file", 1, 0, 'f'},
	{"interval", 1, 0, 'i'},
	{"help", 0, 0, 'h'},
	{"version", 0, 0, 'v'},
	{NULL, 0, 0, 0},
};

void __fail(p_mon m, unsigned char e, int en)
{

	m->err=e;
	if ( e != ERR_CUSTOM ) m->errnum=en;
	
	if ( m->s != -1 ) close(m->s);
	m->s=-1;

	m->fails++;
	m->phase=PHASE_WAIT;
	phase_wait(m, 1);
}

static void phase_wait(p_mon m, int throttle)
{
	/* its gone from failure to success */
	if ( !m->fails && m->lastconn ) {
		if ( !m->state ) {
			status_up(m);
			m->state=1;
		}
	}

	/* Its gone from success to failure */
	if ( m->fails && m->state ) {
		status_down(m);
		m->state=0;
	}

	/* We are already late so do the connect
	* before the next poll. */
	if ( m->lastconn+m->interval <= globals.now ) {
		m->phase=0;
		if ( !throttle ) phase_connect(m);
	}

	/* NOTE on throttle. The throttle parameter is there to
	* stop inifinite loops which block out other monitors such 
	* as failing in the connect state, going straight into the wait
	* state then because the interval is low, straight into connect
	* again, looping forever. Throttle makes sure a poll is done
	* before looping back again after an error.
	*/
}

static void phase_connect(p_mon m)
{
	struct sockaddr_in addr;

	m->lastconn=globals.now;

	/* Do a socket if necessary */
	if ( m->s<0 && !((m->s=async_tcpsocket())>=0) ) {
		__fail(m, ERR_SOCKET, errno);
		return;
	}

	/* Setup the address to connect to */
	addr.sin_family=AF_INET;
	addr.sin_port=htons(m->port);
	addr.sin_addr.s_addr=m->addr;
	
	/* Connect (asynchronous) */
	if ( connect(m->s, (struct sockaddr *)&addr,sizeof(addr))==-1
		&& errno != EINPROGRESS) {
		__fail(m, ERR_CONNECT, errno);
		return;
	}

	/* Stitch in to poll set */
	pf[count].fd=m->s;
	pf[count].events=POLLOUT|POLLHUP|POLLERR;
	pf[count].revents=0;
	m->pindex=count;
	count++;
	m->phase++;
}

static inline void phase_conresult(p_mon m)
{
	int conerr;
	int s=sizeof(conerr);

	/* There is a connect() error status waiting for us! */
	if ( pf[m->pindex].revents & POLLOUT ||
		pf[m->pindex].revents&POLLERR ) {
		if ( getsockopt(m->s, SOL_SOCKET, SO_ERROR, (void *)&conerr, &s) <0 ) {
			__fail(m, ERR_GETSOCKOPT, errno);
			return;
		}

		if ( conerr==0 ) {
			/* Skip straight to next phase so
			* we can get on with I/O */
			m->phase++;
			phase_query(m);
		}else{
			__fail(m, ERR_CONNECT2, conerr);
		}
		return;
	}

	/* If there has been an error, or we have
	* simply been trying for too long
	* reschedule an other attempt for later */
	if ( pf[m->pindex].revents&POLLHUP ) {
		__fail(m, ERR_CONNTIMEOUT, (globals.now-m->lastconn));
		return;
	}
	if ( (globals.now-m->lastconn) >= globals.timeout ) {
		__fail(m, ERR_CONNTIMEOUT, (globals.now-m->lastconn));
		return;
	}
	
	/* If it hasn't yet connected, stitch it back
	* in to the poll FD set for another time round. */
	pf[count].fd=m->s;
	pf[count].events=POLLOUT|POLLHUP|POLLERR;
	pf[count].revents=0;
	m->pindex=count;
	count++;
	return;
}

static void phase_query(p_mon m)
{
	/* In the case of not needing to send a query
	* we always succeed. */
	if ( m->plugin->send_query && 
		!m->plugin->send_query(m->s, m->priv) ) {
		__fail(m, ERR_SEND, errno);
		return;
	}else{
		/* It worked!, poll for reciept of a reply */
		pf[count].fd=m->s;
		pf[count].events=POLLIN|POLLHUP|POLLERR;
		pf[count].revents=0;
		m->pindex=count;
		count++;
		m->phase++;
	}
}

/* PHASE_QUERY has made us poll for input
* so now we should have some */
static inline void phase_recv(p_mon m)
{
	int len;

	if ( pf[m->pindex].revents & POLLIN ) {
		if ( m->plugin->check_results && m->plugin->bufflen) {

			/* buflen will be >0 if plugin->bufflen is */
			if ( (len=recv(m->s, globals.buf, globals.buflen, 0))<0 ) {
				__fail(m, ERR_RECV, errno);
				return;
			}
			
			if ( !(m->plugin->check_results(
				globals.buf, len, m->priv)) ) {
				__fail(m, ERR_CUSTOM, 0);
				return;
			}
		}

		/* If the plugin doesn't have a check_results()
		* function then the fact that the FD is readable
		* is good enough */
		close(m->s);
		m->lastsuccess=globals.now;
		m->fails=0;
		m->s=-1;
		m->phase++;
		phase_wait(m, 0);
		return;
	}

	/* If there has been an error, or we
	* have waited too long, die */
	if ( pf[m->pindex].revents&POLLERR ||
		pf[m->pindex].revents&POLLHUP ) {
		__fail(m, ERR_RECVTIMEOUT, (globals.now-m->lastconn));
		return;
	}
	if ( (globals.now-m->lastconn) >= globals.timeout ) {
		__fail(m, ERR_RECVTIMEOUT, (globals.now-m->lastconn));
		return;
	}
	
	/* Or else have another go */
	pf[count].fd=m->s;
	pf[count].events=POLLIN|POLLHUP|POLLERR;
	pf[count].revents=0;
	m->pindex=count;
	count++;
}

void usage(void)
{
	fprintf(stderr, 
		"Usage:\n"
		"  %s [OPTIONS]\n"
		"\n"
		"Options:\n"
		"  -f, --file     \tSpecifies the location of the config file\n"
		"  -i, --interval \tSpecify the time in seconds between checks (default: 10s)\n"
		"  -h, --help     \tDisplay this usage information\n"
		"\n"
		"  Report bugs to: Gianni Tedesco <gianni@ecsc.co.uk>\n"
		"\n",
		globals.cmd);
}

int parse_args(int argc, char **argv)
{
	int ret=1;
	int c, opti;

	while( (c=getopt_long(argc, argv, "i:f:hv",
		(const struct option *)&opts, &opti))!=-1 ) {
		switch (c)
		{
			case 'i':
				globals.interval=strtoul(optarg, NULL, 10);
				globals.timeout=globals.interval;

				break;
			case 'f':
				globals.config=optarg;
				break;
			case 'h':
				usage();
				exit(0);
			case 'v':
				exit(0);
			default:
				return 0;
		}
		
	}

	return ret;
}

int main ( int argc, char **argv ) 
{
	int pr, i;
	p_mon m;

	printf(VERSION_STRING "\n");
	
	/* Setup the global variables */
	memset(&globals, 0, sizeof(globals));
	globals.cmd=argv[0];
	globals.interval=10;
	globals.timeout=globals.interval;
	globals.config="/etc/lvsmon.conf";

	/* Parse the command line */
	if ( !parse_args(argc, argv) ) {
		usage();
		die_err();
	}

	/* Time out in the nick of time, but
	* no less than 1 second, and no more 
	* than 15 seconds */
	if ( globals.timeout > 15 ) {
		globals.timeout=15;
	}else if ( globals.timeout > 9 ) {
		globals.timeout-=2;
	}else if ( globals.timeout > 2 ) {
		globals.timeout--;
	}
				
	
	/* Hardcoded plugin stuff */
	globals.plugins=&http_plugin;
	globals.plugins->next=&banner_plugin;
	globals.buflen=globals.plugins->bufflen;
	
	/* Parse the config file */
	if ( !config_parse(globals.config) ) {
		die_err();
	}

	/* Check we have something to do */
	if ( num_monitors <= 0 ) {
		fprintf(stderr, "%s: No monitors setup, giving up the will to live\n",
			globals.cmd);
		die_err();
	}

	/* Allocate dynamic resources */
	if ( !(pf=malloc(sizeof(struct pollfd)*num_monitors)) ) {
		fprintf(stderr, "%s: Couldn't allocate pollfd: %s\n",
			globals.cmd, get_err());
		die_err();
	}
	
	if ( globals.buflen > 0 ) {
		if ( !(globals.buf=malloc(globals.buflen)) ) {
			fprintf(stderr, "%s: Couldn't allocate buffer: %s\n",
				globals.cmd, get_err());
			die_err();
		}
	}
	
	/* MAIN LOOP: No more dynamic allocation
	* beyond this point !!! */
	for(;;) {
		count=i=0;
		for(m=monitors; m; m=m->next) {
			/* pretty much everything needs the time.
			* Nothing we do blocks, so we can skip
			* updating it for quite a few iterations.
			* Return isn't checked because it can't fail. */
			if ( !(i%4096) ) {
				time(&globals.now);
			}i++;
			
			switch ( m->phase ) {
				case PHASE_CONNECT:
					phase_connect(m);
					break;
				case PHASE_CONRESULT:
					phase_conresult(m);
					break;
				case PHASE_QUERY:
					phase_query(m);
					break;
				case PHASE_RECV:
					phase_recv(m);
					break;
				case PHASE_WAIT:
					phase_wait(m, 0);
					break;
			}
		}

		if (count) {
			pr=poll(pf, count, 1000); /* 100 ms */
		}else{
			/* Stops us running away with CPU */
			usleep(500000); /* 100 ms */
		}
	}

	/* No need to clean up,
	* we never get here */
	return STATUS_OK;
}
